import json
from datetime import timedelta
from fastapi import FastAPI, Depends, HTTPException, WebSocket, Form, Query,Path
from fastapi.security import OAuth2PasswordBearer, OAuth2PasswordRequestForm
from fastapi.responses import HTMLResponse, RedirectResponse
from pydantic import BaseModel
from auth import hash_password, verify_password, create_access_token, decode_access_token
from typing import Dict

app = FastAPI()
oauth2_scheme = OAuth2PasswordBearer(tokenUrl="login")

# Store active websocket connections
active_connections: Dict[str, WebSocket] = {}

USERS_DB = "users.json"
TASKS_DB = "tasks.json"


class User(BaseModel):
    username: str
    password: str


class Task(BaseModel):
    task: str


# --- JSON helpers ---
def load_json(file):
    try:
        with open(file, "r") as f:
            return json.load(f)
    except FileNotFoundError:
        return {}


def save_json(file, data):
    with open(file, "w") as f:
        json.dump(data, f, indent=4)


def get_current_user(token: str = Depends(oauth2_scheme)):
    payload = decode_access_token(token)
    if not payload:
        raise HTTPException(status_code=401, detail="Invalid or expired token")
    return payload["sub"]


# --- User routes ---
@app.post("/register")
def register(user: User):
    users = load_json(USERS_DB)
    if user.username in users:
        raise HTTPException(status_code=400, detail="Username exists")
    users[user.username] = {
        "username": user.username,
        "password": hash_password(user.password),
    }
    save_json(USERS_DB, users)
    return {"message": "Registered successfully"}


@app.post("/login")
def login(form_data: OAuth2PasswordRequestForm = Depends()):
    users = load_json(USERS_DB)
    user = users.get(form_data.username)
    if not user or not verify_password(form_data.password, user["password"]):
        raise HTTPException(status_code=401, detail="Invalid credentials")

    # create JWT
    token = create_access_token(
        {"sub": form_data.username}, expires_delta=timedelta(minutes=30)
    )

    return {"access_token": token, "token_type": "bearer"}


# ---------- HTML PAGES ----------
@app.get("/", response_class=HTMLResponse)
async def login_page():
    with open("static/login.html", "r") as f:
        return HTMLResponse(f.read())


@app.get("/register", response_class=HTMLResponse)
async def register_page():
    with open("static/register.html", "r") as f:
        return HTMLResponse(f.read())


@app.get("/dashboard", response_class=HTMLResponse)
async def dashboard(token: str = Query(None)):
    if not token:
        return RedirectResponse("/")
    payload = decode_access_token(token)
    if not payload:
        return RedirectResponse("/")
    with open("static/HTML-Websocket.html", "r") as f:
        return HTMLResponse(f.read())


# --- Task routes ---
@app.get("/tasks")
def get_tasks(current_user: str = Depends(get_current_user)):
    tasks_db = load_json(TASKS_DB)
    return tasks_db.get(current_user, [])


@app.post("/add-task")
async def add_task(
    username: str = Form(...), task: str = Form(...), token: str = Query(None)
):
    # Verify token
    payload = decode_access_token(token)
    if not payload or payload["sub"] != username:
        raise HTTPException(status_code=401, detail="Invalid token")

    # Save task
    data = load_json(TASKS_DB)
    if username not in data:
        data[username] = []
    next_id = len(data[username]) + 1
    data[username].append({"id": next_id, "task": task})
    save_json(TASKS_DB, data)

    # Notify user via WebSocket
    if username in active_connections:
        try:
            await active_connections[username].send_text(f"New Task: {task}")
        except Exception:
            active_connections.pop(username, None)

    return {"message": "Task added successfully"}

@app.post("/delete-task")
async def delete_task(
    username: str = Form(...), task_id: int = Form(...), token: str = Query(None)
):
    # Verify token
    payload = decode_access_token(token)
    if not payload or payload["sub"] != username:
        raise HTTPException(status_code=401, detail="Invalid token")

    # Load tasks
    data = load_json(TASKS_DB)
    if username not in data:
        raise HTTPException(status_code=404, detail="No tasks found for this user")

    tasks = data[username]
    new_tasks = [t for t in tasks if t["id"] != task_id]

    if len(tasks) == len(new_tasks):
        raise HTTPException(status_code=404, detail="Task not found")

    data[username] = new_tasks
    save_json(TASKS_DB, data)

    # Notify via WebSocket
    if username in active_connections:
        try:
            await active_connections[username].send_text(f"Task {task_id} deleted")
        except Exception:
            active_connections.pop(username, None)

    return {"message": f"Task {task_id} deleted successfully"}


# --- WebSocket ---
@app.websocket("/ws/{username}")
async def websocket_endpoint(websocket: WebSocket, username: str):
    await websocket.accept()
    active_connections[username] = websocket
    try:
        while True:
            await websocket.receive_text()  # keep alive
            #await websocket.send_text("Websocket Connected")
    except Exception:
        active_connections.pop(username, None)
